<?php


namespace Dmkzwo\ContaoTranslationBundle\Library;

use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Symfony\Component\Yaml\Yaml;

class TranslationHelper
{
    protected $defaultLang;
    protected $languages;

    protected $vendorDir;
    protected $exportFolder = 'translate';
    protected $importFolder = 'translated';
    protected $phpFilesFolder = 'translate_source';
    protected $phpTranslatedFolder = 'translated_source';

    protected $exportDir;
    protected $importDir;
    protected $phpFilesDir;
    protected $phpTranslatedDir;

    protected $timestamp;

    protected $files;
    protected $actLut;


    public function __construct()
    {
        $this->vendorDir = \System::getContainer()->getParameter('dz.translation.vendor_dir');
        $this->defaultLang = \System::getContainer()->getParameter('dz.translation.default_lang');
        $this->languages = explode(',', \System::getContainer()->getParameter('dz.translation.languages'));

        $dataDir = \System::getContainer()->getParameter('dz.translation.data_dir');
        $this->exportDir = TL_ROOT . DIRECTORY_SEPARATOR . $dataDir . DIRECTORY_SEPARATOR . $this->exportFolder;
        $this->importDir = TL_ROOT . DIRECTORY_SEPARATOR . 'files' . DIRECTORY_SEPARATOR . $dataDir . DIRECTORY_SEPARATOR . $this->importFolder;
        $this->phpFilesDir = TL_ROOT . DIRECTORY_SEPARATOR . $dataDir . DIRECTORY_SEPARATOR . $this->phpFilesFolder;
        $this->phpTranslatedDir = TL_ROOT . DIRECTORY_SEPARATOR . $dataDir . DIRECTORY_SEPARATOR . $this->phpTranslatedFolder;


        $this->files = [
            [
                'dir' => 'contao/languages',
                'name' => 'default.php',
                'title' => 'Übergreifend',
            ],
        ];

        $this->files = array_merge($this->files, $this->getSourceFiles());

        if (!is_dir($this->exportDir)) {
            mkdir($this->exportDir, 0755, true);
        }
        if (!is_dir($this->importDir)) {
            mkdir($this->importDir, 0755, true);
        }
        if (!is_dir($this->phpFilesDir)) {
            mkdir($this->phpFilesDir, 0755, true);
        }
        if (!is_dir($this->phpTranslatedDir)) {
            mkdir($this->phpTranslatedDir, 0755, true);
        }

        $this->timestamp = date('YmdHis');
    }

    public function getExportDir()
    {
        return $this->exportDir;
    }

    public function getSourceFiles()
    {
        $absVendorDir = TL_ROOT . DIRECTORY_SEPARATOR . $this->vendorDir;
        $bundles = array_diff(scandir($absVendorDir), array('..', '.'));

        $files = [];

        foreach ($bundles as $bundle) {
            if (!is_dir(TL_ROOT . DIRECTORY_SEPARATOR . $this->vendorDir . DIRECTORY_SEPARATOR . $bundle)) {
                continue;
            }

            $file = [
                'dir' => $this->vendorDir . DIRECTORY_SEPARATOR . $bundle . '/src/Resources/contao/languages',
                'reldir' => $bundle . '/src/Resources/contao/languages',
                'name' => 'default.php',
                'title' => $this->titleFromBundlename($bundle),
                'hastranslation' => $this->hasLanguageFile($bundle)
            ];

            $files[] = $file;
        }

        return $files;
    }

    protected function hasLanguageFile($bundle)
    {
        $languageFile = TL_ROOT . DIRECTORY_SEPARATOR . $this->vendorDir . DIRECTORY_SEPARATOR . $bundle . '/src/Resources/contao/languages/default.php';

        return file_exists($languageFile) && is_file($languageFile);
    }

    public function getTranslationFiles($file)
    {
        $sourceFiles = $this->getSourceFiles();

        $languageFiles = [];
        foreach ($this->languages as $language) {
            $exportFilename = $this->createExportFilename($file['reldir'], $file['name'], $language);
            if ($this->exportFileExists($exportFilename)) {
                $languageFiles[$language] = $exportFilename;
            } else {
                $languageFiles[$language] = false;
            }
        }

        return $languageFiles;
    }

    protected function titleFromBundlename($bundleName)
    {
        return ucfirst(str_replace(
            ['contao-', '-bundle', '-'],
            ['', '', ''],
            $bundleName
        ));
    }

    public function createMissingFiles()
    {
        foreach ($this->files as $file) {
            $sourceDir = $this->createAbsLangDirPath($file['dir'], $this->defaultLang);
            $sourceFile = $sourceDir . '/' . $file['name'];

            // check if default dir exits
            if (!file_exists($sourceDir) || !is_dir($sourceDir)) {
                continue;
            }

            // create lang dirs and files if missing
            foreach ($this->languages as $language) {
                $actDir = $this->createAbsLangDirPath($file['dir'], $language);
                @mkdir($actDir, 0755, true);

                $langFile = $actDir . '/' . $file['name'];
                if (!file_exists($langFile) || !is_file($langFile)) {
                    copy($sourceFile, $langFile);
                    echo 'copy ' . $sourceFile . ' -> ' . $langFile . '<br>';
                }
            }

        }
    }

    public function copySourceFiles()
    {
        $allLanguages = array_merge([$this->defaultLang], $this->languages);

        foreach ($this->files as $file) {
            foreach ($allLanguages as $language) {

                $sourceDir = $this->createAbsLangDirPath($file['dir'], $language);
                $sourceFile = $sourceDir . '/' . $file['name'];

                $targetDir = $this->phpFilesDir . '/' . $file['reldir'] . '/' . $language;
                $targetFile = $targetDir . '/' . $file['name'];

                // check if default dir exits
                if (!file_exists($sourceFile) || !is_file($sourceFile)) {
                    continue;
                }

                @mkdir($targetDir, 0755, true);
                copy($sourceFile, $targetFile);

//                echo $sourceFile;
//                echo '<br>';
//                echo $targetFile;
//                echo '<br>';

            }

        }
    }

    protected function createAbsLangDirPath($dir, $lang)
    {
        return TL_ROOT . DIRECTORY_SEPARATOR . $dir . DIRECTORY_SEPARATOR . $lang;
    }

    protected function createPhpSourcePath($dir, $lang)
    {
        return $this->phpFilesDir . DIRECTORY_SEPARATOR . $dir . DIRECTORY_SEPARATOR . $lang;
    }

    protected function createPhpTargetPath($dir, $lang)
    {
        return $this->phpTranslatedDir . DIRECTORY_SEPARATOR . $dir . DIRECTORY_SEPARATOR . $lang;
    }

    protected function createExportFilename($dir, $name, $lang)
    {
        return str_replace('/', '--', $dir) . '--' . $lang . '--' . str_replace('.php', '.xlsx', $name);
    }

    protected function exportFileExists($filename)
    {
        $file = $this->exportDir . DIRECTORY_SEPARATOR . $filename;

        return file_exists($file) && is_file($file);
    }

    public function exportAll()
    {
        // $this->createMissingFiles();

        foreach ($this->files as $file) {
            $this->exportFile($file);
        }
    }

    public function exportFile($file)
    {
        $defaultLangFile = $this->createPhpSourcePath($file['reldir'], $this->defaultLang) . '/' . $file['name'];

        if (!file_exists($defaultLangFile) || !is_file($defaultLangFile)) {
            return;
        }
//        echo $defaultLangFile;
//        echo '<br>';
//        return;
        $this->actLut = $this->createLutTable($defaultLangFile);

        foreach ($this->languages as $language) {
            $targetLangFile = $this->createPhpSourcePath($file['reldir'], $language) . '/' . $file['name'];

            $targetLut = [];
            if (file_exists($targetLangFile) && is_file($targetLangFile)) {
                $targetLut = $this->createLutTable($targetLangFile);
            }

            $filename = $this->createExportFilename($file['reldir'], $file['name'], $language);
            if (file_exists($filename) && is_file($filename)) {
                unlink($filename);
            }
            $this->createXslx($file['title'], $language, $filename, $targetLut);
//          echo 'create xslx for ' . $targetLangFile . ' (' . $filename . ') <br>';
        }

    }

    protected function createXslx($title, $language, $filename, $targetLut)
    {

        $spreadsheet = new Spreadsheet();

        $spreadsheet->getProperties()
            ->setCreator("Dmkzwo GmbH")
            ->setTitle($title);


        $sheet = $spreadsheet->getActiveSheet();
//        $sheet->setCellValue('A1', $language);

        $this->writeHeader($sheet, $language);
        $this->writeContent($sheet, $targetLut);

        $writer = new Xlsx($spreadsheet);
        $writer->save($this->exportDir . '/' . $filename);


    }


    protected function _createXslx($title, $language, $filename, $sourceFile)
    {

        $spreadsheet = new Spreadsheet();

        $spreadsheet->getProperties()
            ->setCreator("Dmkzwo GmbH")
            ->setTitle($title);


        $sheet = $spreadsheet->getActiveSheet();
//        $sheet->setCellValue('A1', $language);

        $this->writeHeader($sheet, $language);
        $this->writeContent($sheet, $sourceFile);

        $writer = new Xlsx($spreadsheet);
        $writer->save($this->exportDir . '/' . $filename);


    }


    protected function writeHeader($sheet, $language)
    {
        $sheet->setCellValue('A1', "ID (nicht verändern)");
        $sheet->setCellValue('B1', strtoupper($language));
        $sheet->setCellValue('C1', "DE (nicht verändern)");

        foreach (['A', 'B', 'C'] as $letter) {
            $sheet->getStyle($letter . '1')->getFont()->setBold(true);
            $sheet->getStyle($letter . '1')->getBorders()->getBottom()->setBorderStyle(Border::BORDER_THIN);
        }
    }

    protected function writeContent($sheet, $targetLut)
    {
//        $lines = file($sourceFile);

        $rowCounter = 2;

//        $firstLineFound = false;

        foreach ($this->actLut as $key => $label) {
//            if (strpos($line, '$GLOBALS') !== 0) {
//                if (!$firstLineFound) {
//                    continue;
//                }
//
//                $sheet->setCellValue('A' . $rowCounter, "");
//                $sheet->setCellValue('B' . $rowCounter, "");
//
//            } else {
//                if (!$firstLineFound) {
//                    $this->writeEmptyLine($sheet, $rowCounter);
//                    $rowCounter++;
//                }

//                $line = str_replace(['$GLOBALS[\'TL_LANG\']', ";"], ['', ''], $line);

//                $parts = explode(' = ', $line);

            $sheet->setCellValue('A' . $rowCounter, $key);
            $sheet->setCellValue('B' . $rowCounter, $this->cleanLabel($targetLut[$key]));
            $sheet->setCellValue('C' . $rowCounter, $this->cleanLabel($label));

            $firstLineFound = true;
//                echo $line;
//                echo '<br>';
//            }

            $rowCounter++;
        }
    }

    protected function cleanLabel($label)
    {
        $label = trim($label);
        $label = trim($label, "'");
        $label = trim($label, '"');

        $label = str_replace("\'", "'", $label);
        return $label;
    }

    protected function _writeContent($sheet, $sourceFile)
    {
        $lines = file($sourceFile);

        $rowCounter = 2;

        $firstLineFound = false;

        foreach ($lines as $line) {
            if (strpos($line, '$GLOBALS') !== 0) {
                if (!$firstLineFound) {
                    continue;
                }

                $sheet->setCellValue('A' . $rowCounter, "");
                $sheet->setCellValue('B' . $rowCounter, "");

            } else {
                if (!$firstLineFound) {
                    $this->writeEmptyLine($sheet, $rowCounter);
                    $rowCounter++;
                }

                $line = str_replace(['$GLOBALS[\'TL_LANG\']', ";"], ['', ''], $line);

                $parts = explode(' = ', $line);

                $sheet->setCellValue('A' . $rowCounter, $parts[0]);
                $sheet->setCellValue('B' . $rowCounter, trim(trim($parts[1]), "'"));
                $sheet->setCellValue('C' . $rowCounter, $this->actLut[$parts[0]]);

                $firstLineFound = true;
//                echo $line;
//                echo '<br>';
            }

            $rowCounter++;
        }
    }

    protected function writeEmptyLine($sheet, $rowCounter)
    {
        $sheet->setCellValue('A' . $rowCounter, "");
        $sheet->setCellValue('B' . $rowCounter, "");
        $sheet->setCellValue('C' . $rowCounter, "");
    }


    protected function createLutTable($defaultFile)
    {
        $lines = file($defaultFile);

        $lut = [];

        foreach ($lines as $line) {
            if (strpos($line, '$GLOBALS') === 0) {
                $line = str_replace(['$GLOBALS[\'TL_LANG\']', ";"], ['', ''], $line);

                $parts = explode(' = ', $line);

                $lut[$parts[0]] = trim(trim($parts[1]), "'");

            }
        }

        return $lut;
    }


    public function importAll()
    {
        foreach ($this->files as $file) {
            foreach ($this->languages as $language) {
                $actDir = $this->createPhpTargetPath($file['reldir'], $language);
                $langFile = $actDir . '/' . $file['name'];

                if (!is_dir($actDir)) {
                    mkdir($actDir, 0755, true);
                }

                $xlsxFilename = $this->createExportFilename($file['reldir'], $file['name'], $language);
                $xlsxFile = $this->importDir . '/' . $xlsxFilename;

                if (file_exists($xlsxFile) && is_file($xlsxFile)) {
                    echo 'create php for ' . $langFile . ' <br>';
                    $this->createPhp($xlsxFile, $langFile);
                }
            }

        }
    }

    protected function createPhp($xlsxFile, $langFile)
    {
        // backup php file
//        copy($langFile, $langFile . '_backup_' . $this->timestamp);

        $spreadsheet = IOFactory::load($xlsxFile);
        $sheetData = $spreadsheet->getActiveSheet()->toArray(null, true, true, true);
//        var_dump($sheetData);

        $phpContent = '';

        foreach ($sheetData as $row) {
            if (!isset($row['A'])) {
                $phpContent .= "\n";
            } else {
                if (strpos($row['A'], '[') !== 0) {
                    continue;
                }

                $label = str_replace("'", "\'", $row['B']);
                $phpContent .= sprintf('$GLOBALS[\'TL_LANG\']%s = \'%s\'', $row['A'], $label) . ";\n";

//                if (strpos($row['B'], '"') === 0) {
//                    $phpContent .= '$GLOBALS[\'TL_LANG\']' . $row['A'] . ' = ' . $row['B'] . ";\n";
//                } else {
//                    $phpContent .= '$GLOBALS[\'TL_LANG\']' . $row['A'] . ' = ' . "'" . $row['B'] . "';\n";
//                }
            }

        }

        $phpContent = "<?php \n" . $phpContent;

        file_put_contents($langFile, $phpContent);
    }

}
