<?php

declare(strict_types=1);

/*
 * This file is part of [dmkzwo/contao-hello-world-bundle].
 *
 * (c) Dmkzwo
 *
 * @license LGPL-3.0-or-later
 */

namespace Dmkzwo\ContaoDzConextBundle\Classes\Command;

use Contao\CoreBundle\Framework\ContaoFramework;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Question\ConfirmationQuestion;
use Symfony\Component\Filesystem\Filesystem;

class ParametersCommand extends DbBaseCommand
{
    private $rootDir;
    private $framework;

//    private $databaseName;
//    private $databaseUser;
//    private $databasePassword;

    private $createFromTemplate;

    public function __construct(string $rootDir, ContaoFramework $framework)
    {
        $this->rootDir = $rootDir;
        $this->framework = $framework;

        parent::__construct();
    }

    protected function configure(): void
    {
        $this
            ->setName('dmkzwo:parameters')
            ->setDescription('Sets the db parameters in the config/parameters.yml file');

        $this->addOption(
            'create-from-template',
            null,
            InputOption::VALUE_NONE,
            'create parameters.yml from __parameters.yml'
        );
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->framework->initialize();

        $output->writeln([
            '',
            'This sets the database credentials/name in the config/parameters.yml file.',
            '',
            "'--create-from-template' creates/overwrites config/parameters.yml from config/__parameters.yml prior to setting the parameters.",
            '',
        ]);

        $helper = $this->getHelper('question');
        $question = new ConfirmationQuestion('Press any key to continue or ^C to cancel...', true);

        if (!$helper->ask($input, $output, $question)) {
            return 0;
        }

        $this->createFromTemplate = !empty($input->getOption('create-from-template'));

        $this->addDbParamQuestions($input, $output);

        $parametersFile = $this->rootDir.'/config/parameters.yml';
        $parametersTemplateFile = $this->rootDir.'/config/__parameters.yml';

        if ($this->createFromTemplate) {
            if (!file_exists($parametersTemplateFile)) {
                throw new \RuntimeException('File config/__parameters.yml missing');
            }

            $fs = new Filesystem();
            $fs->copy($parametersTemplateFile, $parametersFile, true); // override anyway

            $this->replacePlaceholders($parametersFile);
        } else {
            if (file_exists($parametersFile) && is_file($parametersFile)) {
                $this->replaceData($parametersFile);
            } else {
                throw new \RuntimeException('File config/parameters.yml missing');
            }
        }

        $output->writeln([
            '',
            '------------------',
            'config/parameters.yml created/changed',
            '',
        ]);

        // $output->writeln(file_get_contents($parametersFile));

        return 0;
    }

    protected function replacePlaceholders(string $parametersFile): void
    {
        $content = file_get_contents($parametersFile);

        $content = $this->replaceDbPlaceholders($content);

        file_put_contents($parametersFile, $content);
    }

    protected function replaceData(string $parametersFile): void
    {
        $lines = file($parametersFile);
        foreach ($lines as &$line) {
            if (preg_match('/\s*database_host: /', $line)) {
                $line = '    database_host: '.$this->databaseHost."\n";
            }
            if (preg_match('/\s*database_port: /', $line)) {
                $line = '    database_port: '.$this->databasePort."\n";
            }
            if (preg_match('/\s*database_name: /', $line)) {
                $line = '    database_name: '.$this->databaseName."\n";
            }
            if (preg_match('/\s*database_user: /', $line)) {
                $line = '    database_user: '.$this->databaseUser."\n";
            }
            if (preg_match('/\s*database_password: /', $line)) {
                $line = "    database_password: '".$this->databasePassword."'\n";
            }
        }

        $content = implode('', $lines);

        file_put_contents($parametersFile, $content);
    }
}
