<?php

declare(strict_types=1);

/*
 * This file is part of [dmkzwo/contao-hello-world-bundle].
 *
 * (c) Dmkzwo
 *
 * @license LGPL-3.0-or-later
 */

namespace Dmkzwo\ContaoDzConextBundle\Classes\Command;

use Contao\CoreBundle\Framework\ContaoFramework;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Question\ConfirmationQuestion;
use Symfony\Component\Console\Question\Question;

class PreCommitCommand extends DbBaseCommand
{
    private $rootDir;
    private $framework;

//    private $databaseName;
//    private $databaseUser;
//    private $databasePassword;

    private $databaseCommand;

    private $addToCommit;

    public function __construct(string $rootDir, ContaoFramework $framework)
    {
        $this->rootDir = $rootDir;
        $this->framework = $framework;

        parent::__construct();
    }

    protected function configure(): void
    {
        $this
            ->setName('dmkzwo:precommit')
            ->setDescription('Creates git precommit hook for automated database dump with every commit');

        $this->addOption(
            'add-to-commit',
            null,
            InputOption::VALUE_NONE,
            'add dumpfile to commit'
        );
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->framework->initialize();

        $output->writeln([
            '',
            'This creates/overwrites a git precommit hook for generating an automated database dump with every commit.',
            '',
            "Add '--add-to-commit' to include the dumpfile in the commit",
            '',
        ]);

        $helper = $this->getHelper('question');
        $question = new ConfirmationQuestion('Press any key to continue or ^C to cancel...', true);

        if (!$helper->ask($input, $output, $question)) {
            return 0;
        }

        $this->addToCommit = !empty($input->getOption('add-to-commit'));

        /* Ask for dump command */
        $questionDbCommand = new Question('Mysqldump command [/usr/bin/mysqldump]: ', '/usr/bin/mysqldump');
        $questionDbCommand->setValidator(function ($answer) {
            if (!\strlen($answer) || !file_exists($answer) || !is_file($answer) || !preg_match('/\/mysqldump$/', $answer)) {
                throw new \RuntimeException('mysqldump command not found');
            }

            return $answer;
        });

        $this->databaseCommand = $helper->ask($input, $output, $questionDbCommand);

        $this->addDbParamQuestions($input, $output);

        $templateFile = $this->rootDir.'/vendor/dmkzwo/contao-dz-conext-bundle/src/Resources/file_templates/precommit';
        $targetFile = $this->rootDir.'/.git/hooks/precommit';

        if (!file_exists($templateFile)) {
            throw new \RuntimeException('precommit template file missing');
        }

        $strTemplateContent = file_get_contents($templateFile);

        $strReplacedContent = $this->replacePlaceholders($strTemplateContent);

        file_put_contents($targetFile, $strReplacedContent);

        $output->writeln([
            '',
            '------------------',
            '.git/hook/precommit created',
            '',
            'NOTE: if hook is not executed try the following command:',
            'chmod +x .git/hooks/pre-commit',
            '',
        ]);

        //$output->writeln($strReplacedContent);

        return 0;
    }

//    protected function addDbParamQuestions(InputInterface $input, OutputInterface $output): void {
//        /* Ask for database name */
//        $questionDbName = new Question('Enter the database name: ');
//        $questionDbName->setValidator(function ($answer) {
//            if (!\is_string($answer) || !preg_match('/^[a-zA-Z0-9_]+$/', $answer)) {
//                throw new \RuntimeException('Invalid or empty database name');
//            }
//
//            return $answer;
//        });
//
//        /* Ask for database user */
//        $questionDbUser = new Question('Enter the database user: ');
//        $questionDbUser->setValidator(function ($answer) {
//            if (!\is_string($answer)) {
//                throw new \RuntimeException('Empty database user');
//            }
//
//            return $answer;
//        });
//
//        /* Ask for database password */
//        $questionDbPassword = new Question('Enter the database password: ');
//        $questionDbPassword->setValidator(function ($answer) {
//            if (!\is_string($answer)) {
//                throw new \RuntimeException('Empty database password');
//            }
//
//            return $answer;
//        });
//
//        $helper = $this->getHelper('question');
//
//        $this->databaseName = $helper->ask($input, $output, $questionDbName);
//        $this->databaseUser = $helper->ask($input, $output, $questionDbUser);
//        $this->databasePassword = $helper->ask($input, $output, $questionDbPassword);
//
//    }

    protected function replacePlaceholders(string $content): string
    {
        $content = $this->replaceDbPlaceholders($content);

        $content = str_replace(
            [
                '<database_command>',
            ],
            [
                $this->databaseCommand,
            ],
            $content);

        if ($this->addToCommit) {
            $addToCommitStr = 'printf "Adding database to repository...\n"'."\n".'git add $DBPATH/$DBDUMPFILENAME.sql'."\n";
            $content = str_replace('# <gitadd>', $addToCommitStr, $content);
        } else {
            $content = str_replace('# <gitadd>', '', $content);
        }

        return $content;
    }
}
