<?php

declare(strict_types=1);

/*
 * This file is part of [dmkzwo/contao-docs-bundle].
 *
 * (c) Dmkzwo
 *
 * @license LGPL-3.0-or-later
 */

namespace Dmkzwo\ContaoDocsBundle\EventListener;

use Contao\CoreBundle\Framework\ContaoFramework;
use Dmkzwo\ContaoDocsBundle\Library\DocsHelper;
use Dmkzwo\ContaoDocsBundle\Model\DocsModel;

/**
 * Class InsertTagsListener
 * @Hook("replaceInsertTags")
 */
class InsertTagsListener
{
    private const SUPPORTED_TAGS = [
        'doc_url',
        'doc_title',
        'doc_type',
        'doc_file_url',
        'doc_file_name',
        'doc_file_size',

        'doc_file_namemeta',
        'doc_file_titlemeta',
        'doc_download_meta',
    ];

    /**
     * @var ContaoFramework
     */
    private $framework;

    public function __construct(ContaoFramework $framework)
    {
        die();
        $this->framework = $framework;
    }

    /**
     * @param mixed $cacheValue
     *
     * @return string|false
     */
    public function __invoke(string $tag, bool $useCache, $cacheValue, array $flags)
    {
        die();
        $elements = explode('::', $tag);
        $key = strtolower($elements[0]);

        if (\in_array($key, self::SUPPORTED_TAGS, true)) {
            return $this->replaceInsertTags($key, $elements[1], $flags);
        }

        return false;
    }

    /**
     * @param $file
     */
    protected function getHrFilesize($file): string
    {
        $fileSize = filesize(\System::getContainer()->getParameter('kernel.project_dir').'/'.$file['src']);

        return DocsHelper::humanReadableFilesize($fileSize, 1).'B';
    }

    private function replaceInsertTags(string $insertTag, string $value, array $flags): string
    {
        $this->framework->initialize();

        if (!$value) {
            return '';
        }

        $objDoc = DocsModel::findByPk($value);

        if (!$objDoc) {
            return '';
        }

        $file = DocsHelper::getDocFile($objDoc);

        if (!$file) {
            return '';
        }

        switch ($insertTag) {
            case 'doc_url':
                return DocsHelper::generateDocLinkById($value);

            case 'doc_title':
                return $objDoc->title;

            case 'doc_type':
                return $GLOBALS['TL_LANG']['dz_docs']['doctype'][$value];

            case 'doc_file_url':
                return $file['src'];

            case 'doc_file_name':
                return $file['name'];

            case 'doc_file_size':
                return $this->getHrFilesize($file);

            case 'doc_file_namemeta':
                $suffix = strtoupper(DocsHelper::getSuffix($file['src']));
                $name = sprintf($GLOBALS['TL_LANG']['dz_docs']['save_as'], ucfirst($suffix));
                $hrfileSize = $this->getHrFilesize($file);

                return sprintf('%s (%s)', $name, $hrfileSize);

            case 'doc_file_titlemeta':
                $suffix = strtoupper(DocsHelper::getSuffix($file['src']));
                $hrfileSize = $this->getHrFilesize($file);

                return sprintf('%s (%s, %s)', $objDoc->title, $suffix, $hrfileSize);

            case 'doc_download_meta':
                $suffix = strtoupper(DocsHelper::getSuffix($file['src']));
                $hrfileSize = $this->getHrFilesize($file);

                return sprintf($GLOBALS['TL_LANG']['dz_docs']['save_as'].' (%s)', $suffix, $hrfileSize);
        }

        return '';
    }
}
